/*
 * codec.c
 * Original source from: 昌達 慶仁著，「詳解 画像処理プログラミング」，ソフトバンククリエイティブ株式会社，2008．
 *
 *  Modified on: 2014/06/19 by
 *  The 2nd ARC/CPSY/RECONF High-Performance Computer System Design Contest
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "image.h"
#include "codec.h"
#include "fileio.h"
#include "addressmap.h"

#define N 256
#define NODATA 2100000000

extern int* y_stream  ;
extern int* cr_stream ;
extern int* cb_stream ;
extern int* DCval     ;
extern int* scanline  ;

int	bits = 0;
int	bdata = 0;

// ハフマン木
int l_node[2 * N], r_node[2 * N];
int parent[2 * N];

int Qtable[] = { 16, 11, 10, 16, 24, 40, 51, 61, 12, 12, 14, 19, 26, 58, 60, 55,
		14, 13, 16, 24, 40, 57, 69, 56, 14, 17, 22, 29, 51, 87, 80, 62, 18, 22,
		37, 57, 68, 109, 103, 77, 24, 35, 55, 64, 81, 104, 113, 92, 49, 64, 78,
		87, 103, 121, 120, 101, 72, 92, 95, 98, 112, 100, 103, 99 };

int scan[] = { 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12, 19, 26,
		33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35, 42, 49, 56, 57,
		50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51, 58, 59, 52, 45, 38, 31, 39,
		46, 53, 60, 61, 54, 47, 55, 62, 63 };

/**
 * DCT functions
 */

int DCT8(int *in, int *out) {
	int tmp0, tmp1, tmp2, tmp3;
	int tmp4, tmp5, tmp6, tmp7;
	int u4, u5, u6, u7;

	tmp0 = in[0] + in[7];
	tmp1 = in[1] + in[6];
	tmp2 = in[2] + in[5];
	tmp3 = in[3] + in[4];
	tmp4 = in[3] - in[4];
	tmp5 = in[2] - in[5];
	tmp6 = in[1] - in[6];
	tmp7 = in[0] - in[7];

	out[0] = MULTIPLY( (tmp0 + tmp3), C14) + MULTIPLY((tmp1 + tmp2), C14);
	out[4] = MULTIPLY( (tmp0 + tmp3), C14) - MULTIPLY((tmp1 + tmp2), C14);
	out[2] = MULTIPLY( (tmp1 - tmp2), S18) + MULTIPLY((tmp0 - tmp3), C18);
	out[6] = MULTIPLY(-(tmp1 - tmp2), S38) + MULTIPLY((tmp0 - tmp3), C38);

	u4 = tmp4;
	u5 = MULTIPLY(-tmp5, C14) + MULTIPLY(tmp6, C14);
	u6 = MULTIPLY( tmp5, C14) + MULTIPLY(tmp6, C14);
	u7 = tmp7;

	tmp4 = u4 + u5;
	tmp5 = u4 - u5;
	tmp6 = -u6 + u7;
	tmp7 = u6 + u7;

	out[1] = MULTIPLY( tmp4, S1H) + MULTIPLY(tmp7, C1H);
	out[5] = MULTIPLY( tmp5, S5H) + MULTIPLY(tmp6, C5H);
	out[3] = MULTIPLY(-tmp5, S3H) + MULTIPLY(tmp6, C3H);
	out[7] = MULTIPLY(-tmp4, S7H) + MULTIPLY(tmp7, C7H);

	return 0;
}

int IDCT8(int *in, int *out) {
	int t0, t1, t2, t3, t4, t5, t6, t7;
	int u0, u1, u2, u3, u4, u5, u6, u7;

	t4 = MULTIPLY(in[1], S1H) - MULTIPLY(in[7], S7H);
	t5 = MULTIPLY(in[5], S5H) - MULTIPLY(in[3], S3H);
	t6 = MULTIPLY(in[5], C5H) + MULTIPLY(in[3], C3H);
	t7 = MULTIPLY(in[1], C1H) + MULTIPLY(in[7], C7H);

	u0 = MULTIPLY(in[0], C14) + MULTIPLY(in[4], C14);
	u1 = MULTIPLY(in[0], C14) - MULTIPLY(in[4], C14);
	u2 = MULTIPLY(in[2], S18) - MULTIPLY(in[6], S38);
	u3 = MULTIPLY(in[2], C18) + MULTIPLY(in[6], C38);
	u4 = t4 + t5;
	u5 = t4 - t5;
	u6 = -t6 + t7;
	u7 = t6 + t7;

	t0 = u0 + u3;
	t1 = u1 + u2;
	t2 = u1 - u2;
	t3 = u0 - u3;
	t4 = u4;
	t5 = MULTIPLY(-u5, C14) + MULTIPLY(u6, C14);
	t6 = MULTIPLY( u5, C14) + MULTIPLY(u6, C14);
	t7 = u7;

	out[0] = t0 + t7;
	out[1] = t1 + t6;
	out[2] = t2 + t5;
	out[3] = t3 + t4;
	out[4] = t3 - t4;
	out[5] = t2 - t5;
	out[6] = t1 - t6;
	out[7] = t0 - t7;

	return 0;
}

int tenchi(int *mat, int n) {
	int x, y;
	int tmp;

	for (y = 0; y < n; y++) {
		for (x = 0; x < n; x++) {
			if (x < y) {
				tmp = mat[x + y * n];
				mat[x + y * n] = mat[y + x * n];
				mat[y + x * n] = tmp;
			}
		}
	}
	return 0;
}

/*
 * Color conversions
 */

void convert2ycrcb(ImageData *image_rgb, ImageData *image_ycrcb) {

	int x, y;
	Pixel pix_rgb, pix_ycrcb;

	for (y = 0; y < image_rgb->height; y++) {
		for (x = 0; x < image_rgb->width; x++) {
			getPixel(image_rgb, x, y, &pix_rgb);
			pix_ycrcb.r = MULTIPLY(FIX_0_2989,pix_rgb.r)
					+ MULTIPLY(FIX_0_5866,pix_rgb.g)
					+ MULTIPLY(FIX_0_1145,pix_rgb.b); // Y
			pix_ycrcb.g = MULTIPLY(FIX_0_5000,pix_rgb.r)
					- MULTIPLY(FIX_0_4183,pix_rgb.g)
					- MULTIPLY(FIX_0_0816,pix_rgb.b) + 128; // Cb
			pix_ycrcb.b = -MULTIPLY(FIX_0_1687,pix_rgb.r)
					- MULTIPLY(FIX_0_3312,pix_rgb.g)
					+ MULTIPLY(FIX_0_5000,pix_rgb.b) + 128; // Cr
			setPixel(image_ycrcb, x, y, &pix_ycrcb);

			/*
			 printf("(%d, %d)  r:%d, g:%d, b:%d -> y:%d, cr:%d, cb:%d\n",
			 x, y,
			 pix_rgb.r, pix_rgb.g, pix_rgb.b,
			 pix_ycrcb.r, pix_ycrcb.g, pix_ycrcb.b);
			 */
		}
	}

}

void convert2bmp(ImageData *image_ycrcb, ImageData *image_rgb) {

	int x, y;
	Pixel pix_rgb, pix_ycrcb;
	int pix_y, pix_cr, pix_cb;

	for (y = 0; y < image_ycrcb->height; y++) {
		for (x = 0; x < image_ycrcb->width; x++) {
			getPixel(image_ycrcb, x, y, &pix_ycrcb);
			pix_y = pix_ycrcb.r;
			pix_cr = pix_ycrcb.g - 128;
			pix_cb = pix_ycrcb.b - 128;
			pix_rgb.r = pix_y + MULTIPLY(FIX_1_4022, pix_cr);
			pix_rgb.g = pix_y - MULTIPLY(FIX_0_3456, pix_cb)
					- MULTIPLY(FIX_0_7145, pix_cr);
			pix_rgb.b = pix_y + MULTIPLY(FIX_1_7710, pix_cb);
			setPixel(image_rgb, x, y, &pix_rgb);

			//printf("(%d, %d)  r:%d, g:%d, b:%d\n", x, y, pix_rgb.r, pix_rgb.g, pix_rgb.b);
		}
	}

}


int fgetBitInit() {
	bits = 0;
	bdata = 0;
	return 0;
}

// 1ビット入力
int fgetBit(FILE *fp) {
	unsigned char bbuf;
	int val;

	if (bits == 0) {
		readData(&bbuf, 1, fp);
		bdata = bbuf;
	}
	val = (bdata >> 7) & 1;
	bdata = (bdata << 1) & 0xff;
	bits++;
	if (bits >= 8) {
		bits = 0;
		bdata = 0;
	}
	return val;
}

int fgetNumber(int bits, FILE *fp) {
	int i, res;

	res = 0;
	for (i = 0; i < bits; i++) {
		res = res * 2;
		res += fgetBit(fp);
	}
	return res;
}

int getValue(int n, int datas, FILE *fp) {
	int rt, bit;

	rt = datas - 1;
	while (rt >= n) {
		bit = fgetBit(fp);
		if (bit == 0) {
			rt = l_node[rt];
		} else {
			rt = r_node[rt];
		}
	}
	return rt;
}

void set_stream(int mx, int my, int x, int y, int *stream, int tmp) {
	if (x < 0)
		x = 0;
	if (y < 0)
		y = 0;
	if (x >= mx)
		x = mx - 1;
	if (y >= my)
		y = my - 1;

	stream[mx * y + x] = correctValue(tmp, PIXELMAX);
}

int outputHeader(FILE *fp, int mx, int my, int dep) {
	fputc('D', fp);
	fputc('C', fp);
	fputc('T', fp);
	fputc('3', fp);
	fputLong(mx, fp);
	fputLong(my, fp);
	fputLong(dep, fp);
	return 0;
}

int fputBitInit() {
	bits = 0;
	bdata = 0;
	return 0;
}

// 1ビット出力
int fputBit(int bit, FILE *fp) {
	bdata = (bdata << 1) | bit;
	bits++;
	if (bits >= 8) {
		fputc(bdata, fp);
		bits = 0;
		bdata = 0;
	}
	return 0;
}

// nビット出力
int putBits(FILE *fp, int data, int n) {
	int i;

	for (i = n - 1; i >= 0; i--) {
		fputBit((data >> i) & 1, fp);
	}
	return 0;
}

// 余ったビット出力
int flushBit(FILE *fp) {
	int i;

	for (i = 0; i < 7; i++) {
		fputBit(0, fp);
	}
	return 0;
}

int fputNumber(int num, int bits, FILE *fp) {
	if (num < 0) {
		fputBit(1, fp);
		num = -num;
	} else {
		fputBit(0, fp);
	}
	putBits(fp, num, bits);
	return 0;
}

/*
 * Image Input / Decoding functions
 */

int getImgInfo(int *mx, int *my, int *dep, FILE *fp) {
	unsigned char buf[100];

	readData(buf, 4, fp);
	*mx = getLong(fp);
	*my = getLong(fp);
	*dep = getLong(fp);
	return 0;
}

// 最小頻度の２つを求める
int getMin2(int *hist, int hm, int *d1, int *d2) {
	int i;
	int min;

	min = NODATA - 1;
	*d1 = *d2 = -100;
	for (i = 0; i < hm; i++) {
		if (hist[i] < min) {
			*d1 = i;
			min = hist[i];
		}
	}
	min = NODATA - 1;
	for (i = 0; i < hm; i++) {
		if (i != (*d1) && hist[i] < min) {
			*d2 = i;
			min = hist[i];
		}
	}
	return 0;
}

int makeTree(int *hist, int n) {
	int hm_data;
	int d1, d2, i;

	hm_data = n;

	// 初期化
	for (i = 0; i < 2 + N; i++) {
		l_node[i] = r_node[i] = parent[i] = 0;
	}

	while (1) {
		// 最小頻度の２つを求める
		getMin2(hist, hm_data, &d1, &d2);
		// 最小値となるものがなければ終了
		if (d1 < 0 || d2 < 0)
			break;
		l_node[hm_data] = d1;
		r_node[hm_data] = d2;
		parent[d1] = hm_data;
		parent[d2] = -hm_data;
		hist[hm_data] = hist[d1] + hist[d2];
		hist[d1] = NODATA;
		hist[d2] = NODATA;
		hm_data++;
	}
	return hm_data;
}

int getOrg(int grp, int jun) {
	int mask, mask2;

	mask = (1 << (grp));
	mask2 = mask / 2;
	if (jun < mask2) {
		return jun - mask + 1;
	} else {
		return jun;
	}
}

int decode(FILE *fp, ImageData *img, int *stream) {
  int x, y;
  int lx, ly, loopx, loopy;
  int c[NNN * NNN], b[NNN * NNN], a[NNN * NNN];
  int i, j;
  int ope, val;
  int tmp;
  int grp, jun;
  int datas;
  int DChist[16 * 2], AChist[16 * 16 * 2];
  int mx, my;
  
  //printf("decode fp=%x, img=%x, stream=%x\n", (int) fp, (int)img, (int)stream);

  mx = img->width;
  my = img->height;
  //printf("decode mx=%x, my=%x\n", mx, my);

  loopx = (mx + NNN - 1) / NNN;
  loopy = (my + NNN - 1) / NNN;
  fgetBitInit();
  //initConst();

  // 頻度テーブル読み込み
  for (i = 0; i < 16; i++)
    DChist[i] = getLong(fp);
  for (i = 0; i < 16 * 16; i++)
    AChist[i] = getLong(fp);

  datas = makeTree(DChist, 16);
  for (i = 0; i < loopx * loopy; i++) {
    grp = getValue(16, datas, fp);
    if (grp > 0)
      jun = fgetNumber(grp, fp);
    else
      jun = 0;
    DCval[i] = getOrg(grp, jun);
    //printf("DCval[%d]=%x\n", i, DCval[i]);
  }
  for (i = 1; i < loopx * loopy; i++) {
    DCval[i] += DCval[i - 1];
  }
  //printf("making tree.\n");
  
  datas = makeTree(AChist, 256);
  for (ly = 0; ly < loopy; ly++) {
    //printf("ly=%d\n", ly);
    for (lx = 0; lx < loopx; lx++) {
      //printf("  lx=%d\n", lx);
      c[0] = (int) (DCval[lx + ly * loopx]);
      i = 0;
      while (i < 63) {
	ope = getValue(256, datas, fp);
	if (ope > 0 && ope < 0xf0)
	  val = fgetNumber(ope % 16, fp);
	else
	  val = 0;
	if (ope == 0) {
	  for (; i < 63; i++) {
	    c[scan[i]] = (int) 0;
	  }
	  break;
	} else if (ope < 0x10) {
	  c[scan[i]] = (int) getOrg(ope, val);
	  i++;
	} else if (ope < 0xf0) {
	  for (j = 0; j < ope / 16; j++) {
	    c[scan[i]] = (int) 0;
	    i++;
	  }
	  ope %= 16;
	  c[scan[i]] = (int) getOrg(ope, val);
	  i++;
	} else {
	  ope &= 0xf;
	  for (j = 0; j < 15 + ope; j++) {
	    c[scan[i]] = (int) 0;
	    i++;
	  }
	}
      }
      // 量子化復元
      for (i = 0; i < 64; i++) {
	//if(i%8==7) ;
	c[i] *= Qtable[i];
      }
      for (y = 0; y < NNN; y++) {
	IDCT8(c + y * NNN, b + y * NNN);
      }
      tenchi(b, NNN);
      for (y = 0; y < 8; y++) {
	IDCT8(b + y * NNN, a + y * NNN);
      }
      for (y = 0; y < NNN; y++) {
	for (x = 0; x < NNN; x++) {
	  tmp = (int) (a[x + y * NNN] / 4.0);
	  if (tmp < 0)
	    tmp = 0;
	  if (tmp > 255)
	    tmp = 255;
	  set_stream(mx, my, lx * NNN + x, ly * NNN + y, stream, tmp);
	  //col.r=tmp;
	  //col.g=tmp;
	  //col.b=tmp;
	  //setPixel(img,lx*NNN+x,ly*NNN+y,&col);
	}
      }
    }
  }
  //free(DCval);
  return 0;
}

int getPicture(char* fileData, ImageData *img) {
	char* fname = "dummy";
	int mx, my, dep;
	FILE *fp;

	int x, y;
	Pixel pix;

	Mem_setFileBaseAddress(fileData);
	fp = fopen(fname, "rb");
	if (fp == NULL) {
		printf("File Open Error: %s\n", fname);
		return FALSE;
	}

	// 画像サイズ情報の読み出し
	getImgInfo(&mx, &my, &dep, fp);

	createImage(img, mx, my, 24);

	if (y_stream == NULL)
		printf("memory allocation error\n");
	if (cr_stream == NULL)
		printf("memory allocation error\n");
	if (cb_stream == NULL)
		printf("memory allocation error\n");

	decode(fp, img, y_stream);
	decode(fp, img, cr_stream);
	decode(fp, img, cb_stream);

	for (y = 0; y < my; y++) {
		for (x = 0; x < mx; x++) {
			pix.r = y_stream[mx * y + x];
			pix.g = cr_stream[mx * y + x];
			pix.b = cb_stream[mx * y + x];
			setPixel(img, x, y, &pix);
		}
	}

	fclose(fp);
	return 0;
}

/*
 * Encoding functions
 */

int* get2Dcode(int *p, int *code) {
  int run;
  int run2;

  switch (*p) {
  case EOB:
    *code = 0x00;
    return p + 2;
  case ZERORUN:
    run = *(p + 1);
    if (run >= 15) {
      run2 = run - 15;
      if( run2 > 15 ) {
	run2 = 15;
      }
      //*code = 0xf0 | (run % 16);
      *code = 0xf0 | run2;
      return p + 2;
    } else {
      *code = (run * 16) | (*(p + 2));
      return p + 4;
    }
  default:
    *code = (*p);
    return p + 2;
  }
  return NULL;
}

int getGroup(int n) {
	if (n == 0)
		return 0;
	if (n >= -1 && n <= 1)
		return 1;
	if (n >= -3 && n <= 3)
		return 2;
	if (n >= -7 && n <= 7)
		return 3;
	if (n >= -15 && n <= 15)
		return 4;
	if (n >= -31 && n <= 31)
		return 5;
	if (n >= -63 && n <= 63)
		return 6;
	if (n >= -127 && n <= 127)
		return 7;
	if (n >= -255 && n <= 255)
		return 8;
	if (n >= -511 && n <= 511)
		return 9;
	if (n >= -1023 && n <= 1023)
		return 10;
	if (n >= -2047 && n <= 2047)
		return 11;
	if (n >= -4095 && n <= 4095)
		return 12;
	if (n >= -8191 && n <= 8191)
		return 13;
	if (n >= -16383 && n <= 16383)
		return 14;
	if (n >= -32767 && n <= 32767)
		return 15;

	return 0;
}

int getCode(int c, int *grp, int *jun) {
	int code;
	int mask;

	*grp = getGroup(c);
	if (c < 0)
		code = c + 1;
	else
		code = c;
	mask = (1 << (*grp)) - 1;
	*jun = c & mask;
	if (c < 0)
		(*jun)--;

	return code;
}

int get_stream(int mx, int my, int x, int y, int *stream) {

	if (x < 0)
		x = 0;
	if (y < 0)
		y = 0;
	if (x >= mx)
		x = mx - 1;
	if (y >= my)
		y = my - 1;

	return stream[mx * y + x];
}

int sgni(int x) {
	if (x < 0)
		return -1;
	if (x > 0)
		return 1;
	return 0;
}

int absi(int x) {
	if (x < 0)
		return -x;
	return x;
}

int scanDCT(int *in, int *out)
{
  int i, n;
  int c;
  int run = 0;
  int grp, jun;
  
  n = 0;
  for (i = 0; i < 63; i++) {
    c = in[scan[i]];
    if (c != 0) {
      if (run > 0) {
	out[n++] = ZERORUN;
	out[n++] = run;
      }
      getCode(c, &grp, &jun);
      out[n++] = grp;
      out[n++] = jun;
      run = 0;
    } else {
      run++;
    }
  }
  if (run > 0) {
    out[n++] = EOB;
    out[n++] = 0;
  }
  return n;
}

// ハフマン符号出力
int outputEncode(int val, int end, FILE *fp)
{
	int c;
	int code[100];
	int now, i;
	int s1, a1;
	c = 0;
	now = val;
	while (1) {
		s1 = sgni(parent[now]);
		a1 = absi(parent[now]);
		if (s1 < 0)
			code[c++] = 1;
		else
			code[c++] = 0;
		if (a1 == end - 1)
			break;
		now = a1;
	}
	for (i = c - 1; i >= 0; i--) {
		fputBit(code[i], fp);
	}
	return 0;
}

int encode(FILE *fp, int mx, int my, int *stream) {
  int x, y, i;
  int lx, ly, loopx, loopy;
  int r_re[NNN * NNN], b[NNN * NNN], c[NNN * NNN];
  int dct_res[NNN * NNN];
  int DChist[16 * 2], AChist[16 * 16 * 2];
  int *dc_p;
  int *np, sres;
  int *p;
  int grp, jun;
  int datas;
  int bef, tmp;
  int code;
  
  loopx = (mx + NNN - 1) / NNN;
  loopy = (my + NNN - 1) / NNN;
  
  fputBitInit();
  //initConst();
  
  np = scanline;
  dc_p = DCval;
  for (ly = 0; ly < loopy; ly++) {
    for (lx = 0; lx < loopx; lx++) {
      // バッファ作成
      for (y = 0; y < NNN; y++) {
	for (x = 0; x < NNN; x++) {
	  //getPixel(img,lx*NNN+x,ly*NNN+y,&col);
	  //r_re[x+y*NNN]=(double)(col.r);
	  //r_re[x+y*NNN]=(int)stream[(ly*NNN+y)*mx+lx*NNN+x];
	  r_re[x + y * NNN] = get_stream(mx, my, lx * NNN + x,
					 ly * NNN + y, stream);
	  //printf("%d\n", r_re[x+y*NNN]);
	}
      }
      
      /* 2次元DCT */
      for (y = 0; y < 8; y++) {
	DCT8(r_re + y * 8, b + y * 8);
      }
      tenchi(b, 8);
      for (y = 0; y < 8; y++) {
	DCT8(b + y * 8, c + y * 8);
      }
      for (i = 0; i < NNN * NNN; i++) {
	dct_res[i] = c[i] * 2 / NNN;
	// 量子化
	dct_res[i] /= Qtable[i];
      }

      /* DC成分の抽出 */
      *dc_p = dct_res[0];
      dc_p++;
      /* AC成分のランレングス符号化 */
      sres = scanDCT(dct_res, np);
      
      np += sres;
    }
  }
  // 直流成分の出力
  // 差分値を求める
  bef = (*DCval);
  for (p = DCval + 1; p < dc_p; p++) {
    tmp = (*p) - bef;
    bef = (*p);
    (*p) = tmp;
  }
  // 頻度テーブル生成
  for (i = 0; i < 16; i++)
    DChist[i] = 0;
  for (p = DCval; p < dc_p; p++) {
    getCode(*p, &grp, &jun);
    DChist[grp]++;
  }
  for (i = 0; i < 16 * 16; i++)
    AChist[i] = 0;
  for (p = scanline; p < np;) {
    p = get2Dcode(p, &code);
    AChist[code]++;
  }
  for (i = 0; i < 16; i++)
    fputLong(DChist[i], fp);
  for (i = 0; i < 16 * 16; i++)
    fputLong(AChist[i], fp);
  // ハフマン符号化（木生成）
  datas = makeTree(DChist, 16);
  // ハフマン符号化（出力）
  for (p = DCval; p < dc_p; p++) {
    getCode(*p, &grp, &jun);
    outputEncode(grp, datas, fp);
    if (grp > 0)
      putBits(fp, jun, grp);
  }
  // 交流成分の出力
  datas = makeTree(AChist, 256);
  for (p = scanline; p < np;) {
    p = get2Dcode(p, &code);
    outputEncode(code, datas, fp);
    if (code > 0 && code < 0xf0) {
      putBits(fp, *(p - 1), code & 0xf);
    }
  }
  flushBit(fp);
  return 0;
}

int writePicture(char* fileData, ImageData *img) {
	FILE *fpo;
	char* fname = "dummy";
	int x, y;
	int mx, my, dep;
	int len;

	Pixel pix;

	mx = img->width;
	my = img->height;
	dep = 24;

	Mem_setFileBaseAddress(fileData);
	if ((fpo = fopen((char*) fname, "wb")) == NULL)
		return FALSE;

	outputHeader(fpo, mx, my, dep);

	if (y_stream == NULL) {
		printf("memory allocation error\n");
	}
	if (cr_stream == NULL) {
		printf("memory allocation error\n");
	}
	if (cb_stream == NULL) {
		printf("memory allocation error\n");
	}

	for (y = 0; y < my; y++) {
		for (x = 0; x < mx; x++) {
			getPixel(img, x, y, &pix);
			y_stream[y * mx + x] = pix.r;
			cr_stream[y * mx + x] = pix.g;
			cb_stream[y * mx + x] = pix.b;
		}
	}

	encode(fpo, mx, my, y_stream);
	encode(fpo, mx, my, cr_stream);
	encode(fpo, mx, my, cb_stream);

	len = ftell(fpo);
	fclose(fpo);
	return len;
}
